/*
* This file is part of 3DzzD http://dzzd.net/.
*
* Released under LGPL
*
* 3DzzD is free software: you can redistribute it and/or modify
* it under the terms of the GNU Lesser General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* 3DzzD is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with 3DzzD.  If not, see <http://www.gnu.org/licenses/>.
*
* Copyright 2005 - 2009 Bruno Augier
*/
 
import java.awt.*;
import java.applet.*;

import net.dzzd.access.*;
import net.dzzd.*;
import net.dzzd.utils.*;



/**
 * FirstAnimation
 * 
 * This sample show how to perform simple displacement & rotation on object :
 */
public class FirstAnimation extends Applet implements Runnable
{
	public static final long serialVersionUID = 0x00000001;
	IRender3D render;
	IScene3D scene;
	boolean run=false;

	public void start()
	{
		//Ask 3DzzD factory for a fresh Scene3D
		this.scene=DzzD.newScene3D();	
		
		//Create a Scene3D loader and link it to a 3DS file
		IScene3DLoader loader=DzzD.newScene3DLoader();
		loader.loadScene3D(this.getCodeBase().toString()+"3D/","CUBE.3DS");
		
		//Add the loader to the scene
		this.scene.setScene3DLoader(loader);
		
		//Wait until all object & texture are loaded
		while(this.scene.getNbMonitoredSceneObject()!=0)
		{
		 this.scene.updateMonitoredSceneObjects();	
		 DzzD.sleep(10);
		}
		
		/*
		 * Set the active camera in the 3d scene
		 * 
		 * We use a camera that is inside the 3ds file
		 *
		 * 3DzzD always provide a default camera that you can set using :
		 *  this.scene.setCurrentCamera3DById(0);
		 */
		this.scene.setCurrentCamera3DByName("Camera01");
		
		//Ask 3DzzD factory for a software 3D Render
		this.render=DzzD.newRender3D(this.getClass(),"SOFT",null);
		
		//Add the Render3D canvas to the Applet Panel
		this.setLayout(null);
		this.add(this.render.getCanvas());
		
		//Set the Render3D size
		this.render.setSize(this.getSize().width,this.getSize().height,1);
		
		//Set Camera Aspect ratio to 1:1
		this.scene.getCurrentCamera3D().setZoomY(((double)this.render.getWidth())/((double)this.render.getHeight()));	
		
		//Tell the Render3D wich camera it should use to render
		this.render.setCamera3D(this.scene.getCurrentCamera3D());
		
			
		//Create a main Thread
		Thread mainThread=new Thread(this);
		this.run=true;
		mainThread.start();
	}
	
	/**
	 * Tell the main thread that it can stop running when Applet is destroyed
	 */
	public void destroy()
	{
		this.run=false;
	}
	
	/**
	 * Here is the mainThread run method
	 * called by mainThread.start(); (inded this method is started in a different Thread)
	 */
	public void run()
	{
		while(this.run)
		{
			//Render a single frame
			this.renderSingleFrame();
			Thread.yield();
			
			
			/**
			 * Retrieve the cube in the scene (the mesh name is "Box01"
			 *  each time a scene is loaded, 3DzzD create a root object 
			 *  with the same name of the loaded file, this object can be used 
			 *  to perfom operation on all object loaded in the same file
			 *  in this sample the object name created as root is "CUBE.3DS"
			 */
			IMesh3D cube=this.scene.getMesh3DByName("Box01");
			
			/**
			 * Retreive the rotation of this object
			 *
			 * NOTE : rotation is retrieved in the object parent space
			 *
			 * 
			 * to retrieve object rotation in world use getAxis().getRotationXZY() method after 
			 *   having set the object to world space
			 *
			 * same to retrieve rotation in current camera space after 
			 *  having set the object to camera space
			 */
			IPoint3D cubeRot=cube.getRotation();
			cubeRot.setX(4.0*MathX.sin(System.currentTimeMillis()*Math.PI/2000));
			
			/**
			 * Retreive the position of this object
			 *
			 * NOTE : position is retrieved in the object parent space
			 * 
			 * to retrieve object position in world use getAxis() method after 
			 *   having set the object to world space
			 *
			 * same to retrieve position in current camera space after 
			 *  having set the object to camera space
			 */
			 IPoint3D cubePos=cube.getPosition();
			 cubePos.setX(30.0*MathX.sin(System.currentTimeMillis()*Math.PI/2000));
			
				
		}
		
	}
	
	
	
	public void renderSingleFrame()
	{
		//Set the scene to world space
		this.scene.setScene3DObjectToWorld();
		
		//Set the scene to active camera space
		this.scene.setScene3DObjectToCamera();
		
		//Tell the 3D render to compute & draw the frame
		this.render.renderScene3D(this.scene);
	}
}
